/*-------------------------------------------------------------------

    FCM.CPP

    24.9.2002

    Ismo Krkkinen

    FCM and RFCM. Plus versions that return normal partitionings.

-------------------------------------------------------------------*/

#define VersionNumber "v0.1.2"

#include <float.h>
#include <stdio.h>
#include <time.h>

#include "matrix.hpp"
#include "fcm.h"
#include "gla.h"

extern "C" {

#include "cb.h"
#include "interfc.h"
#include "memctrl.h"
#include <string.h>
#include "cb_util.h"
#include "DistCrit/criteria.h"
#include "DistCrit/distance.h"

}

static void PrintOutput(int Iter, float Error, float Best, int Improved,
    int Output)
{
    switch (Output) {
    case 0: break;
    case 1:
	if (Improved) PrintMessage("%i\t%#.6g\n", Iter, Best);
	break;
    default:
	PrintMessage("%i\t%#9.6g\t%#.6g\n", Iter, Best, Error);
	break;
    }
}

static void PrintTrace(int Print, int ID, int Groups, int Iter, float Error)
{
    if (Print) ErrorMessage("%i\t%i\t%i\t%e\n", ID, Groups, Iter, Error);
}

static void FixCentroidFrequencies(CODEBOOK* CB, PARTITIONING* PA) {
    for (int k = 0; k < BookSize(CB); ++k) VectorFreq(CB, k) = CCFreq(PA, k);
}

int RandomizedFuzzyCMeans(
    TRAININGSET* TS, CODEBOOK* CB, FuzzyPartitioning* FPA,
    CriterionInfo* CI, float* FinalError, int Solutions, int FCMIterations,
    int Output, int Trace, int TraceID)
{
    CODEBOOK CBnew;
    PARTITIONING PAnew;
    FuzzyPartitioning* FPAnew;
    int k, Improved, Sum;
    float Value;
    if (!strstr(CB->GenerationMethod, "RFCM ")) {
	if (*(CB->GenerationMethod)) AddGenerationMethod(CB, " + ");
	AddGenerationMethod(CB, "RFCM ");
	AddGenerationMethod(CB, VersionNumber);
    }
    if (BookSize(TS) == BookSize(CB)) {
        CopyCodebook(TS, CB);
        FPA->UpdateMemberships(CB, ciCriterionDistance(CI));
	return 0;
    }
    CreateNewCodebook(&CBnew, BookSize(CB), TS);
    CreateNewPartitioning(&PAnew, TS, BookSize(CB));
    FPAnew = new FuzzyPartitioning(TS, BookSize(CB));
    *FinalError = 1e30;
    Sum = 0;
    for (k = 0; k < Solutions; ++k) {
        GenerateRandomCodebook(TS, &CBnew);
        Sum += FuzzyCMeans(TS, &CBnew, FPAnew, FCMIterations, CI);
        FPAnew->Crispify(&PAnew);
        FixCentroidFrequencies(&CBnew, &PAnew);
	Value = ciEvaluate(CI, &CBnew, &PAnew);
	Improved = *FinalError > Value;
	if (Improved) {
	    CopyCodebook(&CBnew, CB);
            *FPA = *FPAnew;
	    *FinalError = Value;
	    PrintTrace(Trace, TraceID, BookSize(CB), Sum, *FinalError);
	}
	if (Output) PrintOutput(k + 1, Value, *FinalError, Improved, Output);
    }
    PrintTrace(Trace, TraceID, BookSize(CB), Sum, *FinalError);
    FreeCodebook(&CBnew);
    FreePartitioning(&PAnew);
    delete FPAnew;
    return Sum;
}

int FuzzyCMeans(TRAININGSET* TS, CODEBOOK* CB, FuzzyPartitioning* FPA,
    int FCMIterations, CriterionInfo* CI)
{
    if (!strstr(CB->GenerationMethod, "FCM ")) {
	if (*(CB->GenerationMethod)) AddGenerationMethod(CB, " + ");
	AddGenerationMethod(CB, "FCM ");
	AddGenerationMethod(CB, VersionNumber);
    }
    if (BookSize(TS) == BookSize(CB)) {
        CopyCodebook(TS, CB);
        FPA->UpdateMemberships(CB, ciCriterionDistance(CI));
	return 0;
    }
    return EmbeddedFCM(TS, CB, FPA, FCMIterations, CI);
}

static int FuzzyCentroidUpdate(TRAININGSET* TS, CODEBOOK* CB,
    FuzzyPartitioning* FPA)
{
#if 1
    static Matrix Sums, USums, V;
    Sums.Resize(BookSize(CB), VectorSize(CB));
    USums.Resize(BookSize(CB), 1);
    V.Resize(VectorSize(TS), 1);
    Sums.Fill(0.0f);
    USums.Fill(0.0f);
    for (int n = 0; n < BookSize(TS); ++n) {
        float Freq = VectorFreq(TS, n);
        ConstRowIterator Iter = FPA->RowBegin(n);
        for (int k = 0; k < Sums.Rows(); ++k, ++Iter) {
            float u = *Iter;
            u *= u * Freq;
            USums[k] += u;
            RowIterator SumsIter = Sums.RowBegin(k);
            for (int d = 0; d < Sums.Cols(); ++d, ++SumsIter)
                *SumsIter += u * VectorScalar(TS, n, d);
        }
    }
    int MaxDiff = 0;
    for (int k = 0; k < BookSize(CB); ++k) {
        float InvUSum;
        if (USums[k] > 1e-6f) InvUSum = 1.0f / USums[k];
        else InvUSum = 1e6f;
        RowIterator SumsIter = Sums.RowBegin(k);
        for (int d = 0; d < VectorSize(CB); ++d, ++SumsIter) {
            int NewScalar = (int)(*SumsIter * InvUSum + 0.5f);
            int Diff = abs(VectorScalar(CB, k, d) - NewScalar);
            if (Diff > MaxDiff) MaxDiff = Diff;
            VectorScalar(CB, k, d) = NewScalar;
        }
    }
#elif 0
    static Matrix Sums, USums, V;
    Sums.Resize(BookSize(CB), VectorSize(CB));
    USums.Resize(BookSize(CB), 1);
    V.Resize(VectorSize(TS), 1);
    Sums.Fill(0.0f);
    USums.Fill(0.0f);
    for (int n = 0; n < BookSize(TS); ++n) {
        float Freq = VectorFreq(TS, n);
        //for (int d = 0; d < V.Rows(); ++d)
            //V[d] = float(VectorFreq(TS, n)) * VectorScalar(TS, n, d);
        for (int k = 0; k < Sums.Rows(); ++k) {
            float u = (*FPA)(n, k);
            u *= u * Freq;
            USums[k] += u;
            for (int d = 0; d < Sums.Cols(); ++d)
                Sums(k, d) += u * VectorScalar(TS, n, d);
        }
    }
    int MaxDiff = 0;
    for (int k = 0; k < BookSize(CB); ++k) {
        float InvUSum;
        if (USums[k] > 1e-6f) InvUSum = 1.0f / USums[k];
        else InvUSum = 1e6f;
        for (int d = 0; d < VectorSize(CB); ++d) {
            int NewScalar = (int)(Sums(k, d) * InvUSum);
            int Diff = abs(VectorScalar(CB, k, d) - NewScalar);
            if (Diff > MaxDiff) MaxDiff = Diff;
            VectorScalar(CB, k, d) = NewScalar;
        }
    }
#else
    int MaxDiff = 0;
    float* Sum = new float[VectorSize(TS)];
    for (int k = 0; k < BookSize(CB); ++k) {
        for (int n = 0; n < VectorSize(TS); ++n) Sum[n] = 0;
        float USum = 0;
        for (int n = 0; n < BookSize(TS); ++n) {
            float u = (*FPA)(n, k);
            u *= u * VectorFreq(TS, n);
            USum += u;
            for (int d = 0; d < VectorSize(TS); ++d)
                Sum[d] += u * VectorScalar(TS, n, d);
        }
        if (USum < 1e-6f) USum = 1e6f;
        else USum = 1.0f / USum;
        for (int d = 0; d < VectorSize(TS); ++d) {
            int NewScalar = (int)(Sum[d] * USum);
            int Diff = abs(VectorScalar(CB, k, d) - NewScalar);
            if (Diff > MaxDiff) MaxDiff = Diff;
            VectorScalar(CB, k, d) = NewScalar;
        }
    }
    delete[] Sum;
#endif
    return MaxDiff;
}

int EmbeddedFCM(TRAININGSET* TS, CODEBOOK* CB, FuzzyPartitioning* FPA,
    int FCMIterations, CriterionInfo* CI)
{
    int k;
    if (!FCMIterations) FCMIterations = -1;
    for (k = 0; k != FCMIterations; ++k) {
        FPA->UpdateMemberships(CB, ciCriterionDistance(CI));
        int MaxDiff = FuzzyCentroidUpdate(TS, CB, FPA);
        if (!MaxDiff || float(MaxDiff) / TS->MaxValue < 1e-6) break;
    }
    return k;
}


int CrispRandomizedFuzzyCMeans(
    TRAININGSET* TS, CODEBOOK* CB, PARTITIONING* PA,
    CriterionInfo* CI, float* FinalError, int Solutions, int FCMIterations,
    int FinishWithGLA, int Output, int Trace, int TraceID)
{
    CODEBOOK CBnew;
    PARTITIONING PAnew;
    FuzzyPartitioning* FPAnew;
    int k, Improved, Sum;
    float Value;
    if (!strstr(CB->GenerationMethod, "RFCM ")) {
        if (*(CB->GenerationMethod)) AddGenerationMethod(CB, " + ");
        AddGenerationMethod(CB, "RFCM ");
        AddGenerationMethod(CB, VersionNumber);
    }
    if (BookSize(TS) == BookSize(CB)) {
        PutAllInOwnPartition(TS, PA);
        GenerateOptimalCodebook(TS, CB, PA);
        return 0;
    }
    CreateNewCodebook(&CBnew, BookSize(CB), TS);
    CreateNewPartitioning(&PAnew, TS, BookSize(CB));
    FPAnew = new FuzzyPartitioning(TS, BookSize(CB));
    *FinalError = 1e30;
    Sum = 0;
    int* Changes = 0;
    if (FinishWithGLA >= 0) Changes = (int*)malloc(sizeof(int) * BookSize(CB));
    for (k = 0; k < Solutions; ++k) {
        GenerateRandomCodebook(TS, &CBnew);
        Sum += FuzzyCMeans(TS, &CBnew, FPAnew, FCMIterations, CI);
        FPAnew->Crispify(&PAnew);
        FixCentroidFrequencies(&CBnew, &PAnew);
        if (FinishWithGLA >= 0) {
            memset(Changes, 1, sizeof(int) * BookSize(CB));
            EmbeddedGLA(TS, &CBnew, &PAnew, FinishWithGLA, CI, Changes);
        }
        Value = ciEvaluate(CI, &CBnew, &PAnew);
        Improved = *FinalError > Value;
        if (Improved) {
            CopyCodebook(&CBnew, CB);
            CopyPartitioning(&PAnew, PA);
            *FinalError = Value;
            PrintTrace(Trace, TraceID, BookSize(CB), Sum, *FinalError);
        }
        if (Output) PrintOutput(k + 1, Value, *FinalError, Improved, Output);
    }
    PrintTrace(Trace, TraceID, BookSize(CB), Sum, *FinalError);
    free(Changes);
    FreeCodebook(&CBnew);
    FreePartitioning(&PAnew);
    delete FPAnew;
    return Sum;
}

int CrispFuzzyCMeans(TRAININGSET* TS, CODEBOOK* CB, PARTITIONING* PA,
    int FCMIterations, CriterionInfo* CI, int FinishWithGLA)
{
    FuzzyPartitioning* FPA = Fuzzify(TS, PA, 0);
    FCMIterations = FuzzyCMeans(TS, CB, FPA, FCMIterations, CI);
    Crispify(FPA, PA);
    FixCentroidFrequencies(CB, PA);
    delete FPA;
    if (FinishWithGLA >= 0) {
        int* Changes = (int*)malloc(sizeof(int) * BookSize(CB));
        memset(Changes, 1, sizeof(int) * BookSize(CB));
        EmbeddedGLA(TS, CB, PA, FinishWithGLA, CI, Changes);
        free(Changes);
    }
    return FCMIterations;
}

int CrispEmbeddedFCM(TRAININGSET* TS, CODEBOOK* CB, PARTITIONING* PA,
    int FCMIterations, CriterionInfo* CI)
{
    FuzzyPartitioning* FPA = Fuzzify(TS, PA, 0);
    FCMIterations = EmbeddedFCM(TS, CB, FPA, FCMIterations, CI);
    Crispify(FPA, PA);
    FixCentroidFrequencies(CB, PA);
    delete FPA;
    return FCMIterations;
}


