/*-------------------------------------------------------------------

    RLS.C

    5.10.2001

    Ismo Krkkinen

    Randomized local search. Re-implementation to take new distance
    and stopping criteria into account.

-------------------------------------------------------------------*/

#define VersionNumber "v0.2.8"

#include <float.h>
#include <stdio.h>
#include <time.h>

#include "rls.h"
#include "cb.h"
#include "interfc.h"
#include "memctrl.h"
#include "random.h"
#include <string.h>
#include "cb_util.h"


static void PrintOutput(int Iter, float Error, float Best, int Improved,
    int Output)
{
    switch (Output) {
    case 0: break;
    case RLS_IMPROVED:
	if (Improved) PrintMessage("%i\t%#.6g\n", Iter, Best);
	break;
    default:
	PrintMessage("%i\t%#9.6g\t%#.6g\n", Iter, Best, Error);
	break;
    }
}

static void PrintTrace(int Print, int ID, int Groups, int Iter, float Error) {
    if (Print) ErrorMessage("%i\t%i\t%i\t%e\n", ID, Groups, Iter, Error);
}

int RandomizedLocalSearch(TRAININGSET* TS, CODEBOOK* CB, PARTITIONING* PA,
    CriterionInfo* CI, float* FinalError,
    Iterations* Iter, int GLAIterations,
    int SwapMethod, int Output, int Trace, int TraceID)
{
    CODEBOOK CBnew;
    PARTITIONING PAnew;
    int IterationNumber, k, Improved, From, To, TotalGLAIterations;
    float Value;
    int ReturnValue = EVERYTHING_OK;
    int *Changes, *Freq2IndexMap;
    if (!strstr(CB->GenerationMethod, "RLS ")) {
	if (*(CB->GenerationMethod)) AddGenerationMethod(CB, " + ");
	AddGenerationMethod(CB, "RLS ");
	AddGenerationMethod(CB, VersionNumber);
    }
    if (BookSize(TS) == BookSize(CB)) {
	PutAllInOwnPartition(TS, PA);
	GenerateOptimalCodebookGeneral(TS, CB, PA, MSE);
	return ReturnValue;
    }
    Freq2IndexMap = FrequencyToIndexMap(TS);
    Changes = allocate(sizeof(int) * BookSize(CB));
    CreateNewCodebook(&CBnew, BookSize(CB), TS);
    CreateNewPartitioning(&PAnew, TS, BookSize(CB));
    IterationNumber = -1;
    *FinalError = ciEvaluate(CI, CB, PA);
    PrintTrace(Trace, TraceID, BookSize(CB), IterationNumber, *FinalError);
    TotalGLAIterations = 0;
    do {
	IterationNumber++;
	CopyCodebook(CB, &CBnew);
	CopyPartitioning(PA, &PAnew);
        memset(Changes, 0, sizeof(int) * BookSize(CB));
	SwapCentroid(CI, TS, &CBnew, &PAnew, SwapMethod, &From, &To,
	    Freq2IndexMap);
	ciLocalPartition(CI, &CBnew, &PAnew, Changes, To);
	Improved = 1;
	for (k = 0; (k < GLAIterations) && Improved; k++) {
	    TotalGLAIterations++;
	    ciCalculateOptimalCentroids(CI, &CBnew, &PAnew, Changes);
	    Improved = ciPartitionOptimally(CI, &CBnew, &PAnew, Changes);
	}
	ciCalculateOptimalCentroids(CI, &CBnew, &PAnew, Changes);
	Value = ciEvaluate(CI, &CBnew, &PAnew);
	Improved = *FinalError > Value;
	if (Improved) {
	    CopyCodebook(&CBnew, CB);
	    CopyPartitioning(&PAnew, PA);
	    *FinalError = Value;
	    PrintTrace(Trace, TraceID, BookSize(CB), IterationNumber + 1,
	        *FinalError);
	}
        if (ExitRequested()) {
	    ReturnValue = EXIT_REQUEST_HONOURED;
	    break;
        }
	if (Output) PrintOutput(IterationNumber, Value, *FinalError,
	    Improved, Output);
    } while (!itStop(Iter, *FinalError));
    PrintTrace(Trace, TraceID, BookSize(CB), IterationNumber, *FinalError);
    if (Output) PrintMessage(
	"Average GLA iterations per RLS iteration (max = %i): %f.\n",
	GLAIterations, TotalGLAIterations / (float) (IterationNumber + 1));
    deallocate(Freq2IndexMap);
    FreeCodebook(&CBnew);
    FreePartitioning(&PAnew);
    deallocate(Changes);
    return ReturnValue;
}

